// *****************************************************************************
//
// Copyright (c) 2022, Pleora Technologies Inc., All rights reserved.
//
// *****************************************************************************

#ifndef __PVTRIGGERSELECTOR_H__
#define __PVTRIGGERSELECTOR_H__


#include <string>

#include <PvVirtualDeviceLib.h>



///
/// \brief This class implements a triggering mechanism
///

class PV_VIRTUAL_DEVICE_API IPvTriggerSelector
{
public:
    enum State
    {
        State_Disabled,
        State_Enabled,
        State_Armed,
        State_Fired,
        State_OneShot
    };

    virtual ~IPvTriggerSelector() {}

    virtual void Start( const uint32_t aHeight ) = 0;
    virtual void Stop() = 0;
    virtual void FireTrigger() = 0;
    virtual void Rearm() = 0;

    virtual void SetMode( bool aIsEnabled ) = 0;
    virtual bool GetMode() const = 0;
    virtual void SetSource( uint32_t aSource ) = 0;
    virtual uint32_t GetSource() const = 0;
    virtual std::string GetName() const = 0;
    virtual State GetState() const = 0;
    virtual bool IsEnabled() const = 0;
};


// Default base class.
// Used internally when TriggerMode[TriggerSelector] is set to OFF.
// Also defines base behavior for most of trigger selectors.
class PV_VIRTUAL_DEVICE_API PvTriggerSelectorDefault
    : public IPvTriggerSelector
{
public:
    PvTriggerSelectorDefault( const char * const aName = "N/A" )
        : mSource( 0 )
        , mState( State_Disabled )
    {
        mName = new std::string( aName );
    }
    virtual ~PvTriggerSelectorDefault() { PVDELETE( mName ); }

    void Start( const uint32_t aHeight ) override { PVUNREFPARAM( aHeight ); }
    void Stop() override {}
    void FireTrigger() override {}
    void Rearm() override {};

    void SetMode( const bool aIsEnabled ) override { SetState( aIsEnabled ? State_Enabled : State_Disabled ); }
    bool GetMode() const override { return IsEnabled(); }
    void SetSource( const uint32_t aSource) override { mSource = aSource; }
    uint32_t GetSource() const override { return mSource; }
    std::string GetName() const override { return *mName; }
    bool IsEnabled() const override { return State_Disabled != mState; }

protected:

    void SetState( const State aState ) { mState = aState; }
    State GetState() const override { return mState; }

private:
    std::string *mName;
    uint32_t mSource;
    volatile State mState;

};


///
/// Implements AcquisitionStart behavior for TriggerSelector.
///
/// AcquisitionStart requires to be triggered once, and remains triggered until stopped.
///
class PV_VIRTUAL_DEVICE_API PvTriggerSelectorAcquisitionStart
    : public PvTriggerSelectorDefault
{
public:
    PvTriggerSelectorAcquisitionStart( const char * const aName = "AcquisitionStart" )
        : PvTriggerSelectorDefault( aName ) {}
    virtual ~PvTriggerSelectorAcquisitionStart() {}

    void Start( const uint32_t aHeight ) override
    {
        PVUNREFPARAM( aHeight );

        if ( State_Enabled == GetState() )
        {
            SetState( State_Armed );
        }
    }

    void Stop() override
    {
        if ( State_Disabled != GetState() )
        {
            SetState( State_Enabled );
        }
    }

    void FireTrigger() override
    {
        if ( State_Armed == GetState() )
        {
            SetState( State_OneShot );
        }
    }

};


///
/// Implements FrameStart behavior for TriggerSelector.
///
/// Behaves just like AcquisitionStart, but is re-armed on a frame basis.
///
class PV_VIRTUAL_DEVICE_API PvTriggerSelectorFrameStart
    : public PvTriggerSelectorAcquisitionStart
{
public:
    PvTriggerSelectorFrameStart( const char * const aName = "FrameStart" )
        : PvTriggerSelectorAcquisitionStart( aName ) {}
    virtual ~PvTriggerSelectorFrameStart() {}

    void Rearm() override
    {
        if ( GetState() == State_Fired )
        {
            SetState( State_Armed );
        }
    }

    void FireTrigger() override
    {
        if ( State_Armed == GetState() )
        {
            SetState( State_Fired );
        }
    }

};


///
/// Implements LineStart behavior for TriggerSelector.
///
/// Behaves like FrameStart, but moves to Fired state once all lines of a frame are received.
///
class PV_VIRTUAL_DEVICE_API PvTriggerSelectorLineStart
    : public PvTriggerSelectorFrameStart
{
public:
    PvTriggerSelectorLineStart( const char * const aName = "LineStart" )
        : PvTriggerSelectorFrameStart( aName )
        , mTrigCount( 1 )
        , mTrigReload( 1 )
    {}
    ~PvTriggerSelectorLineStart() {}

    void Start( const uint32_t aHeight ) override
    {
        if ( State_Enabled == GetState() )
        {
            mTrigReload = aHeight;
            mTrigCount = mTrigReload;
            SetState( State_Armed );
        }
    }

    void FireTrigger() override
    {
        if ( State_Armed == GetState() )
        {
            --mTrigCount;
            if ( !mTrigCount )
            {
                SetState( State_Fired );
            }
        }
    }

    void Rearm() override
    {
        if ( GetState() == State_Fired )
        {
            mTrigCount = mTrigReload;
            SetState( State_Armed );
        }
    }

private:
    uint32_t mTrigCount;
    uint32_t mTrigReload;
};

#endif // __PVTRIGGERSELECTOR_H__
